bl_info = {
    "name": "Camera Angle Straightener",
    "author": "InDizains",
    "version": (1, 1),
    "blender": (4, 0, 0),
    "location": "View3D > Object Context Menu",
    "description": "Straighten camera angles + auto-switch to camera tab reliably",
    "category": "Object",
}

import bpy
import math

_last_active_object = None  # Store last object to detect change

# ---------- CAMERA STRAIGHTEN OPERATOR ----------
def straighten_angle(angle_rad):
    angle_deg = math.degrees(angle_rad)
    straight_deg = round(angle_deg / 90) * 90
    return math.radians(straight_deg)

def straighten_camera_rotation(context):
    obj = context.active_object
    if obj and obj.type == 'CAMERA':
        rot = obj.rotation_euler
        obj.rotation_euler = (
            straighten_angle(rot.x),
            straighten_angle(rot.y),
            straighten_angle(rot.z)
        )
        return {'FINISHED'}
    return {'CANCELLED'}

class OBJECT_OT_straighten_camera_rotation(bpy.types.Operator):
    """Straighten Camera Rotation Angles"""
    bl_idname = "object.straighten_camera_rotation"
    bl_label = "Straighten Camera Angle"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        return straighten_camera_rotation(context)

def draw_straighten_button(self, context):
    obj = context.active_object
    if obj and obj.type == 'CAMERA':
        self.layout.operator(OBJECT_OT_straighten_camera_rotation.bl_idname)


# ---------- CAMERA TAB AUTO-SWITCH ----------
def camera_tab_watcher():
    global _last_active_object

    current = bpy.context.active_object
    if current != _last_active_object:
        _last_active_object = current
        if current and current.type == 'CAMERA':
            for area in bpy.context.screen.areas:
                if area.type == 'PROPERTIES':
                    for space in area.spaces:
                        if space.type == 'PROPERTIES':
                            space.context = 'DATA'
                            break
    return 0.5  # Check twice a second


# ---------- REGISTER / UNREGISTER ----------
def register():
    bpy.utils.register_class(OBJECT_OT_straighten_camera_rotation)
    bpy.types.VIEW3D_MT_object_context_menu.append(draw_straighten_button)
    bpy.app.timers.register(camera_tab_watcher)

def unregister():
    bpy.utils.unregister_class(OBJECT_OT_straighten_camera_rotation)
    bpy.types.VIEW3D_MT_object_context_menu.remove(draw_straighten_button)

if __name__ == "__main__":
    register()
